/* LIBINV.C
 *
 * A Leaf Incorporating Biochemistry Exhibiting Reflectance
 * and Transmittance Yields
 *
 * Inverted model
 *
 * Written by: Terence Dawson
 * University of Southampton
 *
 * Started : 8th March 1995
 * Finished:
 * Last modified:       10th November 1995
 *                      16th November 1996
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>

/* Existing LIBERTY routines... */

void startup(void);
void refrac(void);
void para_rad(void);
void vert_rad(void);
void tot_ref(void);
void eval_me(void);
void eval_mi(void);
void leaf_params(void);
void eval_x(void);
void calc_M(void);
void calc_T(void);
void calc_R(void);
void tidyup(void);

/* New subroutines for inversion... */

void invtrans(void);
void inv_m(void);
void inv_coeff(void);

/* Defining variables and parameters... */

double PI;
double N0, N1, alpha, beta, in_angle, vert_r, para_r, refl;
double critical, me, mi, coeff, M, T, x, R, Alt_R, tinv, xinv, minv, coinv;
float D, xu; /* sphere diameter and air gap parameters */


main()
{
startup();
leaf_params();
tidyup();
}

/*---------------------------------------------------------------------*/
/* Start-up screen and initialisation of variables and parameters etc */

void startup(void)
{
printf("\nThe LIBERTY Leaf Model inversion program\n\n");
printf("Version 1.0A\n\n");
printf("\nDeveloped by :\nTerence Dawson\n");
printf("Department of Geography\n");
printf("University of Southampton\n\n");
printf("(c) Terence Dawson, February 1996\n\n");
para_r=0;
vert_r=0;
PI = 180 * (atan(1)/45);
}

/*---------------------------------------------------------------------*/

/* end signature */
void tidyup(void)
{
   printf("\nAll files closed - Normal Termination.\n");
   printf("coefficient data has been written to the file libinv.out\n");
}

/*---------------------------------------------------------------------*/

/* Applying the index of refraction */
void refrac(void)
{
/* angle of incident light */
in_angle = 59;
/* Index of refraction */
N0 = 1.0;
alpha = in_angle * PI/180;
beta = asin((N0/N1) * sin(alpha));
/* printf("Alpha: %f Beta: %f\n",alpha,beta); */
}

/*---------------------------------------------------------------------*/

/* working out the horizontal (parallel to plane) component of reflected */
/* radiation (Snells Law of Refraction) */
void para_rad(void)
{
	/* printf("Testing coeficients of reflection...\n"); */
	para_r = (tan(alpha - beta))/(tan(alpha + beta));
	/* printf("Parallel: %f",para_r); */
}

/*--------------------------------------------------------------------*/

/* working out the vertical component of reflected radiation...*/
void vert_rad(void)
{
	vert_r = -(sin(alpha-beta))/(sin(alpha+beta));
	/* printf(" Vertical: %f\n",vert_r); */
}

/*--------------------------------------------------------------------*/

/* working out the total reflected radiation...*/
void tot_ref(void)
{
	double plus, dif;

	plus = alpha + beta;
	dif = alpha - beta;

	refl =  0.5 * ( ((sin(dif)*sin(dif))/(sin(plus)*sin(plus))) +
		((tan(dif)*tan(dif))/(tan(plus)*tan(plus))) );
}


/*-------------------------------------------------------------------------*/

/* The evaluation of the regular reflectance for diffuse incident */
/* radiation me for angles of alpha between 0 and PI/2  */
void eval_me(void)
{
   int a;
   double width;

   me = 0;
   width = PI/180;

   for (a = 1; a <= 90; a++) {
	alpha = a * PI/180;
	beta = asin(N0/N1 * sin(alpha));
	tot_ref();
	me = me + (refl * sin(alpha) * cos (alpha) * width);
	}
   me=me*2;
   /* printf("me : %f\n",me); */
}

/*-------------------------------------------------------------------------*/

void eval_mi(void)
{
   int a;
   double mint,width;

   mi = 0;
   mint = 0;
   width = PI/180;
   critical = asin(N0/N1)*180/PI;
   /* printf("Critical angle: %f\n",critical); */
   for (a = 1; a <= (critical); a++) {
	alpha = a * PI/180;
	beta = asin((N0/N1) * sin(alpha));
	tot_ref();
	mint = mint + (refl * sin(alpha) * cos (alpha) * width);
	}
   mi = (1 - (sin(critical*PI/180))*(sin(critical*PI/180))) + (2*mint);
   /* printf("mi : %f\n",mi); */
}


/*-----------------------------------------------------------------------*/

void leaf_params(void)
{
	FILE *infile, *outfile;
	int i, numwav;
	float baseline;
	float wavl[500],refstd[500];
	char filename[20];


	/* Open liberty.out for writing data...*/
	if ((outfile = fopen("libinv.out","w")) == NULL) {
	    printf("Could not open libinv.out\n");
	    }
	rewind(outfile);
	/* write the name of the variables to outfile */
	fprintf(outfile,"W R coinv\n");
	/* Now to work out the value of M, the total radiation reaching
	the surface after one pass through the sphere */


	/* For inversion, input empirical data...*/
	printf("Enter full filename of reflectance data to invert :  ");
	scanf("%s",filename);
	printf("\nFull filename: %s\n",filename);
	/* strcat("\\",filename); */
	if ((infile = fopen(filename,"rt")) == NULL) {
	    printf("Could not open reflectance data file\n");
	    exit(0);
	    }
	printf("\nOpening file for reading...\n");
	/* Ok so far, Now to read in the absorption data... */
	rewind(infile);
	i=0;
	printf("Reading empirical reflectance data...\n");
	while (!feof(infile)) {
	     fscanf(infile,"%f %f\n",&wavl[i],&refstd[i]);
	     if (fmod(i,50)==0) printf(".");
	     i=i+1;
	}
	printf("\n");
	fclose(infile);
	printf("reflectance data from %f nm to %f nm\n",wavl[0],wavl[i-1]);

	numwav = i - 1;

	/*** sphere diameter ***/
	printf("Average leaf cell diameter : ");
	scanf("%f",&D);

	/*** gap function between cells ***/
	printf("Leaf intercellular air gap determinant : ");
	scanf("%f",&xu);

	/* set baseline high */

	baseline=1.0;


	for (i = 0; i <= numwav; i++) {

	  /* change of refractive index over wavelength... */

	  N1=1.4891-(0.0005*i);
	  refrac();
	  para_rad();
	  vert_rad();
	  eval_me();
	  eval_mi();
	  /*
	  calc_M();
	  calc_T();
	  eval_x();
	  calc_R(); */

	  R=refstd[i];
	  invtrans();
	  inv_m();
	  inv_coeff();
	  if (coinv > 0) { if (coinv <= baseline) baseline=coinv; };
	  printf("%f %f %f\n",wavl[i],R,coinv);
	  fprintf(outfile,"%f %f %f\n",wavl[i],R,coinv);
	  /* now to update refstd[i] to the inverted spectra... */
	  refstd[i]=coinv;
	  }
	fclose(outfile);

	printf("\nlinear baseline shift : %f\n",baseline);

}

/*-----------------------------------------------------------------------*/

void calc_M(void)
{
	  M = (2/(coeff * coeff))*(1-(coeff+1)*exp(-coeff));
}

/*-----------------------------------------------------------------------*/

void calc_T(void)
{
	  T = ((1-mi) * M)/(1-(mi * M));
}

/*-----------------------------------------------------------------------*/

void calc_R(void)
{
	double a,b,c,q,m, next_R;
	int iterations;

	a = (me * T) + (x * T) - me - T - (x * me * T);
	b = 1 + (x * me * T) - (2 * x * x * me * me * T);
	c = (2 * me * x * x * T) - (x * T) - (2 * x * me);
	/* we need to check for negative square root! */
	m = (b*b) - (4 * a * c);
	if (m < 0) m = 0;
	q = -0.5 * (b + sqrt(m));
	Alt_R = c / q;

	/* There are some problems with the above hence I'm */
	/* trying out the legendary Newton-Raphson method ... */

	R = 0.5;   /* initial guess */
	for (iterations = 1; iterations < 50; iterations++) {
		/* next_R = Alt_R - (a*(Alt_R*Alt_R) + (b*Alt_R) + c)/((2*a*Alt_R)+b); */
		/* problems hence simple iterative method... */
		next_R = -(a*(R*R)+c)/b;
		/* printf("alt_r: %f\n", next_R); */
		R = next_R;
		}


}

/*-----------------------------------------------------------------------*/

void eval_x(void)
{
	x = xu / (1 - (1 - (2*xu)) * T);
}

/*-----------------------------------------------------------------------*/

void invtrans(void)
{
	double a,b,c,d,q;
	double nex,func,diriv;
	int iterations;

	/* This is a horrendous polynomial function  so we'll */
	/* try out the Newton-Raphson iteration method...     */
	q=(1/(1-(2*xu)));
	d = (q*(R*R)*xu)-(me*q*(R*R)*xu);
	c= R-(me*(R*R))-(q*(R*R))+(q*xu)-(q*me*R*xu)-(q*(R*R)*xu)+(q*me*(R*R))+(q*me*xu*(R*R));
	b=(q*(R*R))-(2*me)-q-(2*q*me*xu)+(q*me*R)+(2*q*(me*me)*R*xu)-(q*me*(R*R));
	a = (2*q*me)-(2*q*(me*me)*R);
	xinv = 0.5;   /* initial guess */
	for (iterations = 1; iterations < 20; iterations++) {
		func=(a*xinv*xinv*xinv)+(b*xinv*xinv)+(c*xinv)+d;
		diriv=(3*a*xinv*xinv)+(2*b*xinv)+c;
		nex=xinv-(func/diriv);
		/*  printf("%f:",nex); */
		if (fabs(nex-xinv) < 0.000001) {
		break; /* no point in wasting cpu time! */
		}
		xinv = nex;
		}

	tinv=(xinv-xu)/(xinv-(2*xinv*xu));

}


/*-----------------------------------------------------------------------*/

void inv_m(void)
{
	minv=tinv/(1-mi+(mi*tinv));
}


/*-----------------------------------------------------------------------*/

void inv_coeff(void)
{
/* based upon ole Newton-Raphson again... */

   double newcoeff,func,diriv;
   int iterations;

   coinv = 0.5;   /* initial guess */
   /* For very low values of R, we need to give a larger estimate */
   if (R<=0.04) { coinv=2;
		if (R<=0.012) coinv=20;
		}
   for (iterations = 1; iterations < 50; iterations++) {
	func=((minv/2)*(coinv*coinv))+(coinv*exp(-coinv))+(exp(-coinv))-1;
	diriv=(2*(minv/2)*coinv)-(coinv*exp(-coinv));
	newcoeff=coinv-(func/diriv);
	/* printf("invtst: %f %f\n",newcoeff,coinv); */
	if (newcoeff <= 0) newcoeff=2;
	if (fabs(newcoeff-coinv) < 0.000001) {
		break; /* no point in wasting cpu time! */
		}
	coinv = newcoeff;
		}
	/* finally, divide by average cell diameter D */
	coinv = coinv/D;
}

/*-----------------------------------------------------------------------*/
