pro p2_idl

; Widget program for the Prospect Model version2
; Jacquemoud et al., 1996
;
; Glenn Newnham
; Remote Sensing and Satellite Research Group
; Curtin University of Technology
; Perth, Western Australia
; email: glenn.newnham@adelaide.edu.au
;
; Created 06.03.01
; Last modified: 20.03.01
;
;
; This code was first developed as an interactive spectral
; matching tool for spectrophotometer data. It has been cut
; down so that it just shows the Prospect model output but
; can easily be adapted to include input from any ascii files
; for the purpose of visual comparison. Please send comments
; to the above email address.
;
;
;**********************************************************************

tlb=widget_base(column=1, title='Prospect Model Version 2', $
	mbar=bar, uvalue={n:1.0, cab:20.0, cw:0.002, cp:0.001, cc:0.001, $
	ang:1, measured:fltarr(3,2151), mfile:'', modelled:fltarr(3,421)})
draw=widget_draw(tlb, graphics_level=1, xsize=540, ysize=300)
base1=widget_base(tlb, /row)
nslide=cw_fslider2(base1, EVENT_PRO='nslide_event', title='Leaf Thickness', /frame, $
	minimum=1, maximum=5, xsize=170, format='(G8.2)', uvalue=0.0, /edit)
cabslide=cw_fslider2(base1, EVENT_PRO='cabslide_event', title='Chlorophyll Concentration (ug/cm2)', $
/frame, minimum=0, maximum=100, xsize=170, value=20, format='(G8.2)', uvalue=0.0, /edit)
cwslide=cw_fslider2(base1, EVENT_PRO='cwslide_event', title='Leaf Water Thickness (cm)', /frame, $
	minimum=0, maximum=0.050, xsize=170, value=0.002, format='(G10.4)', uvalue=0.0, /edit)
base2=widget_base(tlb, /row)
cpslide=cw_fslider2(base2, EVENT_PRO='cpslide_event', title='Protein Content (ug/cm2)', /frame, $
	minimum=0.0, maximum=0.01, xsize=170, value=0.001, format='(G10.4)', uvalue=0.0, /edit)
ccslide=cw_fslider2(base2, EVENT_PRO='ccslide_event', title='Cellulose + Lignin (ug/cm2)', /frame, $
	minimum=0.0, maximum=0.01, xsize=170, value=0.001, format='(G10.4)', uvalue=0.0, /edit)
angslide=cw_fslider2(base2, EVENT_PRO='angslide_event', title='Max. Incident Angle (degrees)', /frame, $
	minimum=0, maximum=90, xsize=170, value=1, format='(G8.2)', uvalue=0.0, /edit)
plot=widget_button(tlb, value='Save Computer Graphics Metafile (.cgm)')
exit=widget_button(tlb, value='Exit')

widget_control, tlb, /realize
widget_control, draw, get_value=winindex    ;determine the draw window index
set_plot, 'win'
wset, winindex						;set the draw window as the current device
device, decomposed=0
loadct, 12

xmanager, 'p2wid', tlb, event='p2wid_event', /no_block

end

;**************************************************
pro nslide_event, event
on_error, 2
widget_control, /hourglass
widget_control, event.top, get_uvalue=struct
widget_control, event.id, get_value=x
struct.n=x
modelled=call_function('prospect2',struct)
struct.modelled=modelled
widget_control, event.top, set_uvalue=struct
end

pro cabslide_event, event
on_error, 2
widget_control, /hourglass
widget_control, event.top, get_uvalue=struct
widget_control, event.id, get_value=x
struct.cab=x
modelled=call_function('prospect2',struct)
struct.modelled=modelled
widget_control, event.top, set_uvalue=struct
end

pro cwslide_event, event
on_error, 2
widget_control, /hourglass
widget_control, event.top, get_uvalue=struct
widget_control, event.id, get_value=x
struct.cw=x
modelled=call_function('prospect2',struct)
struct.modelled=modelled
widget_control, event.top, set_uvalue=struct
end

pro cpslide_event, event
on_error, 2
widget_control, /hourglass
widget_control, event.top, get_uvalue=struct
widget_control, event.id, get_value=x
struct.cp=x
modelled=call_function('prospect2',struct)
struct.modelled=modelled
widget_control, event.top, set_uvalue=struct
end

pro ccslide_event, event
on_error, 2
widget_control, /hourglass
widget_control, event.top, get_uvalue=struct
widget_control, event.id, get_value=x
struct.cc=x
modelled=call_function('prospect2',struct)
struct.modelled=modelled
widget_control, event.top, set_uvalue=struct
end

pro angslide_event, event
on_error, 2
widget_control, /hourglass
widget_control, event.top, get_uvalue=struct
widget_control, event.id, get_value=x
struct.ang=x
modelled=call_function('prospect2',struct)
struct.modelled=modelled
widget_control, event.top, set_uvalue=struct
end

pro p2wid_event, event
on_error, 2
widget_control, /hourglass
header=''
data=fltarr(3,2151)
data2=fltarr(3,421)
widget_control, event.id, get_value=a
widget_control, event.top, get_uvalue=struct
case a of
'Save Computer Graphics Metafile (.cgm)': begin
	widget_control, event.top, get_uvalue=struct
	data2=struct.modelled

	outfile=dialog_pickfile(title='Specify Output CGM File Name', filter='*.cgm')

	if (outfile ne '') then begin
	set_plot, 'cgm'
	device, file=outfile
	plot, data2(0,*), data2(1,*), yrange=[0,1], xtitle='Wavelength (nm)', $
		ytitle='Reflectance / 1-Transmittance', linestyle=0
	oplot, data2(0,*), 1-data2(2,*), linestyle=0
	device,/close
	set_plot,'win'
	endif

	end
'Exit': begin
	widget_control, event.top, /destroy
	end
endcase
end


;********************************************************



function prospect2, struct

;************************************************************
;
; IDL version of the Prospect Leaf Reflectance Model
;
; Jacquemoud S., Ustin S.L., Verdebout J., Schmuck G., Andreoli G.,
; Hosgood B. (1996), Estimating leaf biochemistry using the PROSPECT
; leaf optical properties model, Remote Sens. Environ., 56:194-202
;
; Jacquemoud S., Baret F. (1990), PROSPECT: a model of leaf optical
; properties spectra, Remote Sens. Environ., 34:75-91.
;
; Glenn Newnham 09.03.00
; last updated 14.04.00
;

valeur=''
file=''
scrap = strarr(1)
data = dblarr(7,421)
wave = dblarr(421)
refra = dblarr(421)
ke = dblarr(421)
kab = dblarr(421)
kw = dblarr(421)
kp = dblarr(421)
kc = dblarr(421)
k = dblarr(421)
tau = dblarr(421)
tav1 = dblarr(421)
tav2 = dblarr(421)

refl = fltarr(421)
tran = fltarr(421)
ref_tran = fltarr(2,421)
modelled = fltarr(3,421)

n=struct.n
cab=struct.cab
cw=struct.cw
cp =struct.cp
cc =struct.cc
deg=struct.ang
mfile=struct.mfile

measured=struct.measured
measw=findgen(421)*5+400
measr=measured(1,*)
meast=measured(2,*)

;read in wavelength, leaf refractive index and specific absorption coefficients

valeur='valeur2.txt'
openr, 1, valeur
readf, 1, scrap
readf, 1, data
 wave=data(0,*)
 refra=data(1,*)
 ke=data(2,*)
 kab=data(3,*)
 kw=data(4,*)
 kp=data(5,*)
 kc=data(6,*)
close, 1


;compute the total leaf absorption coefficient from biochemical components

 k=ke+(cab*kab+cw*kw+cp*kp+cc*kc)/n

;compute the total leaf transmission coefficient for leaf internal material

tau = call_function('leaf',k)

;find the average interface transmittance for isotropic light

alpha = 90*(!pi/180)
tav1 = call_function('avg' ,refra,alpha)

;find the average interface transmittance for the solid angle incident on the leaf surface

alpha = deg*(!pi/180)
tav2 = call_function('avg' ,refra ,alpha)

;compute the reflectance and transmittance for a single layer
;Allen et al, 1969
;Jacquemoud and Baret, 1990

x1=1-tav1
x2=tav1^2*tau^2*(refra^2-tav1)
x3=tav1^2*tau*refra^2
x4=refra^4-tau^2*(refra^2-tav1)^2
x5=tav2/tav1
x6=x5*(tav1-1)+1-tav2
r=x1+x2/x4
t=x3/x4
ra=x5*r+x6
ta=x5*t


;compute the reflectance and transmittance for n leaf layers
;Stokes, 1862

delta=(t^2-r^2-1)^2-4*r^2
alfa=(1+r^2-t^2+sqrt(delta))/(2*r)
beta=(1+r^2-t^2-sqrt(delta))/(2*r)

va=(1+r^2-t^2+sqrt(delta))/(2*r)
vb=sqrt(beta*(alfa-r)/(alfa*(beta-r)))

s1=ra*(va*vb^(n-1)-va^(-1)*vb^(-(n-1))) $
	+(ta*t-ra*r)*(vb^(n-1)-vb^(-(n-1)))
s2=ta*(va-va^(-1))
s3=va*vb^(n-1)-va^(-1)*vb^(-(n-1)) $
	-r*(vb^(n-1)-vb^(-(n-1)))

refl=s1/s3
tran=s2/s3


;plots of results
!p.multi=[0,1,1]
plot, [400, 2500],[0, 1], /NODATA, xtitle='Wavelength (nm)', $
	ytitle='reflectance / 1-transmittance', color=255
oplot, measw, refl, linestyle=0, color=32
oplot, measw, 1-tran, linestyle=0, color=32

modelled(0,*)=wave
modelled(1,*)=refl
modelled(2,*)=tran

return, modelled

end


;*****************************************************************************

function leaf, k
;
; Transmission coefficient for the leaf internal material
; Allen et al, 1969, eq.14
;
; Uses the NAG Fortran routine s13aaf
;
; Glenn Newnham 09.03.00
; last updated 14.04.00
;

i=0
x=dblarr(421)
y=dblarr(421)
exint = dblarr(421)
tau = dblarr(421)
deg = 0.0
alpha = 0.0

;compute transmission coefficient tau (Allen et al. 1969, eq.14)

for i=0,420 do begin

case 1 of

	(k(i) le 0): tau(i)= 1

   	(k(i) gt 0) and (k(i) le 4): begin
		x=(0.5*k(i))-1

		y=(((((((((((((((-3.60311230482612224d-13 $
		*x+3.46348526554087424d-12)*x-2.99627399604128973d-11) $
        *x+2.57747807106988589d-10)*x-2.09330568435488303d-9) $
        *x+1.59501329936987818d-8)*x-1.13717900285428895d-7) $
        *x+7.55292885309152956d-7)*x-4.64980751480619431d-6) $
        *x+2.63830365675408129d-5)*x-1.37089870978830576d-4) $
        *x+6.47686503728103400d-4)*x-2.76060141343627983d-3) $
        *x+1.05306034687449505d-2)*x-3.57191348753631956d-2) $
        *x+1.07774527938978692d-1)*x-2.96997075145080963d-1

        y=(y*x+8.64664716763387311d-1)*x+7.42047691268006429d-1
        exint=y-alog(k(i))
		tau(i) = (1.0-k(i))*exp(-k(i))+k(i)^2*exint
		end

	(k(i) gt 4) and (k(i) lt 85): begin
		x=14.5/(k(i)+3.25)-1

		y=(((((((((((((((-1.62806570868460749d-12 $
        *x-8.95400579318284288d-13)*x-4.08352702838151578d-12) $
        *x-1.45132988248537498d-11)*x-8.35086918940757852d-11) $
        *x-2.13638678953766289d-10)*x-1.10302431467069770d-9) $
        *x-3.67128915633455484d-9)*x-1.66980544304104726d-8) $
        *x-6.11774386401295125d-8)*x-2.70306163610271497d-7) $
        *x-1.05565006992891261d-6)*x-4.72090467203711484d-6) $
        *x-1.95076375089955937d-5)*x-9.16450482931221453d-5) $
        *x-4.05892130452128677d-4)*x-2.14213055000334718d-3

        y=((y*x-1.06374875116569657d-2)*x-8.50699154984571871d-2)*x $
        +9.23755307807784058d-1
		exint=exp(-k(i))*y/k(i)
	 	tau(i) = (1-k(i))*exp(-k(i))+k(i)^2*exint
		end

	(k ge 85): tau(i) = (1-k(i))*exp(-k(i))

endcase

endfor

return, tau

end


;************************************************************************************

function avg, refra, alpha
;
; Evaluate the average transmittance across an interface between two dielectrics
; for any solid angle of incidence
;
; Stern, 1964
; Allen, 1973
;
; Glenn Newnham 09.03.00
; last updated 14.04.00
;

a=0.0
b=0.0
ds=0.0
b0=dblarr(421)
b1=dblarr(421)
b2=dblarr(421)
ts=dblarr(421)	;parallel polarised transmittance
tp1=dblarr(421)
tp2=dblarr(421)
tp3=dblarr(421)
tp4=dblarr(421)
tp5=dblarr(421)
tp=dblarr(421)	;perpendicular polarised transmittance
tav=dblarr(421)	;total transmittance for the solid angle of incidence

;compute the single interface average transmittance for the solid angle of incidence

a=((refra+1)^2)/2
b=-(((refra^2)-1)^2)/4
ds=sin(alpha)


case 1 of

	(alpha eq 0): tav=4*refra/(refra+1)^2

	(alpha ne 0): begin
		if (alpha eq !pi/2) then begin
			b1=0
		endif else begin
			b1=sqrt((ds^2-(refra^2+1)/2)^2+b)
		endelse

		b2=ds^2-(refra^2+1)/2
		b0=b1-b2

		ts=(b^2/(6*b0^3)+(b/b0)-(b0/2))-(b^2/(6*a^3))-(b/a)+(a/2)

		tp1=-2*(refra^2)*(b0-a)/(refra^2+1)^2
		tp2=-2*(refra^2)*(refra^2+1)*alog(b0/a)/(refra^2-1)^2
		tp3=refra^2*(1/b0-1/a)/2
		tp4=(16*refra^4*(refra^4+1))*alog((2*(refra^2+1)*b0-(refra^2-1)^2)/(2*(refra^2+1) $
			*a-(refra^2-1)^2))/((refra^2+1)^3*(refra^2-1)^2)
		tp5=16*refra^6*(1/(2*(refra^2+1)*b0-(refra^2-1)^2)-1/(2*(refra^2+1) $
			*a-(refra^2-1)^2))/(refra^2+1)^3
		tp=tp1+tp2+tp3+tp4+tp5

 		tav=(ts+tp)/(2*ds^2)
 		end
endcase

return, tav

end
